#!/usr/bin/perl

# search-doaj.pl - query a solr/lucene index of DOAJ content

# Eric Lease Morgan <eric_morgan@infomotions.com>
# January 12, 2009 - version 1.0


# define
use constant SOLR => 'http://localhost:210/solr';
use constant ROWS => 100;
use constant MIN  => 5;

# require
use strict;
use WebService::Solr;

# initalize
my $solr = WebService::Solr->new( SOLR );

# sanity check
my $query = $ARGV[ 0 ];
if ( ! $query ) {

	print "Usage: $0 <query>\n";
	exit;
	
}

# search; get no more than ROWS records and subject facets occuring MIN times
my $response  = $solr->search( $query, { 'rows'           => ROWS,
                                         'facet'          => 'true', 
                                         'facet.field'    => 'facet_subject', 
                                         'facet.mincount' => MIN });

# get the number of hits, and start display
my $hit_count = $response->pager->total_entries;
print "Your search ($query) found $hit_count document(s).\n\n";

# extract subject facets, and display
my %subjects = &get_facets( $response->facet_counts->{ facet_fields }->{ facet_subject } );
if ( $hit_count ) {

	print "  Subject facets: ";
	foreach ( sort( keys( %subjects ))) { print "$_ (" . $subjects{ $_ } . "); " }
	print "\n\n";
	
}
	
# display each hit
my $index = 0;
foreach my $doc ( $response->docs ) {

	# slurp
	my $id        = $doc->value_for( 'id' );
	my $title     = $doc->value_for( 'title' );
	my $publisher = $doc->value_for( 'publisher' );
	my $url       = $doc->value_for( 'url' );
	my @subjects  = $doc->values_for( 'subject' );
	
	# increment
	$index++;
	
	# echo
	print "     record: $index\n";
	print "         id: $id\n";
	print "      title: $title\n";
	print "  publisher: $publisher\n";
	foreach ( @subjects ) { print "    subject: $_\n" }
	print "        url: $url\n";
	print "\n";
	
}

# done 
exit;


sub get_facets {

	# convert array of facet/hit-count pairs into a hash; obtuse
	my $array_ref = shift;
	my %facet;
	my $i = 0;
	foreach ( @$array_ref ) {
	
		my $k = $array_ref->[ $i ]; $i++;
		my $v = $array_ref->[ $i ]; $i++;
		next if ( ! $v );
		$facet{ $k } = $v;
	 
	}
	
	return %facet;
	
}

